/*
 * $Id: mp3.c,v 1.9 2001/02/09 06:10:41 neo Exp $
 * modified to mp3.c by Nathan Hjelm
 */

#include <string.h>
#include <errno.h>

#include <stdlib.h>
#include <stdio.h>
#include <sys/ioctl.h>
#include <sys/types.h>
#include <sys/uio.h>
#include <sys/stat.h>
#include <time.h>
#include <unistd.h>
#include <fcntl.h>

#ifdef linux
#include <byteswap.h>
#include <endian.h>
#elif defined(__FreeBSD__) || defined(__MacOSX__)
#include <machine/endian.h>
#endif

#include "config.h"

#ifdef HAVE_LIBGEN_H
#include <libgen.h>
#endif

#include "rio-cpp.h"
#include "rio-bsd.h"
#include "hexdump.h"

#ifdef HAVE_ID3_H
#include <id3.h>
#endif

/*
 * Decoding MP3 frame headers is discussed here:
 * http://www.mp3-tech.org/programmer/frame_header.html
 *
 * Also see common.c from mpg123
 */

/**
 * Index elements are [MPEG VERSION][LAYER][BITRATEINDEX]
 * MPEG version 1.0 == 0, MPEG version 2.0 and 2.5 == 1
 */
static const int MP3_BITRATES[2][3][16] = {
  /* MPEG Version 1.0 */
  {   
    /* Layer I */
    {0,32,64,96,128,160,192,224,256,288,320,352,384,416,448,},
    /* Layer II */
    {0,32,48,56, 64, 80, 96,112,128,160,192,224,256,320,384,},
    /* Layer III */
    {0,32,40,48, 56, 64, 80, 96,112,128,160,192,224,256,320,}
  },

  /* MPEG Version 2.0 and 2.5 */
  {   
    /* Layer I */
    {0,32,48,56,64,80,96,112,128,144,160,176,192,224,256,},
    /* Layer II */
    {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,},
    /* Layer III */
    {0,8,16,24,32,40,48,56,64,80,96,112,128,144,160,}
  }
};

/**
 * Sampling rate frequency index 
 * 
 * Index elements are [MPEG_VERSION][FREQUENCY_INDEX]
 * MPEG version 1.0 == 3, MPEG version 2.0 == 1, MPEG version 2.5 == 0
 * index 01 == reserved
 */
static const long MP3_FREQS[4][4] = { 
  /* MPEG Version 2.5 */
  {
    11025 , 12000 , 8000, 0  /* 0 == reserved */
  },
  /* Reserved */
  {
    0, 0, 0, 0
  },
  /* MEPG Version 2.0 */
  {
    22050, 24000, 16000, 0  /* 0 == reserved */
  },
  /* MPEG Version 1.0 */
  {
    44100, 48000, 32000, 0  /* 0 == reserved */
  }
};

static const char *MPEG_LAYERS[4] = { "(reserved)", "III", "II", "I"};

#define MPEG_LAYER_I   3
#define MPEG_LAYER_II  2
#define MPEG_LAYER_III 1

static const char *MPEG_VERSION_NAMES[4] = {
  "2.5", "(reserved)", "2.0", "1.0"
};

#define MPEG_VERSION_1        3
#define MPEG_VERSION_2        2
#define MPEG_VERSION_RESERVED 1
#define MPEG_VERSION_25       0

/**
 * Returns the MP3 layer from the provided header.
 */
static int getMP3Layer(unsigned long head) {
  /* Bits (18,17)
     Layer description
     00 - reserved
     01 - Layer III
     10 - Layer II
     11 - Layer I 
  */

  unsigned long mask = (1 << 17) | (1 << 18);
  int layer = (head & mask) >> 17;
  if ( layer == 0 ) {
    //    cerr << "Error, reserved MP3 layer 00 detected" << endl;
    layer = -1;
  }

  return layer;
}

/**
 * Returns the MPEG version from the provided header.
 */
static int getMPEGVersion(unsigned long head) {
  /* Bits (20, 19)
     MPEG Audio version ID
     00 - MPEG Version 2.5 (later extension of MPEG2)
     01 - reserved
     10 - MPEG Version 2 (ISO/IEC 13818-3)
     11 - MPEG Version 1 (ISO/IEC 11172-3) 
                     
     Note: MPEG Version 2.5 was added lately to the MPEG 2 standard. 
     		It is an extension used for very low bitrate files, allowing 
    		the use of lower sampling frequencies. If your decoder does 
    		not support this extension, it is recommended for you to use 
    		12 bits for synchronization instead of 11 bits. 
  */

  unsigned long mask = (1 << 20) | (1 << 19);
  int version = (head & mask) >> 19;

  if ( version == 1 ) {
    //    cerr << "Error: reserved MPEG version id of 01 detected" << endl;
    version = -1;
  }
  return version;
}

/**
 * Returns the MP3 bitrate indexc from the provided header.
 */
static int getMP3BitrateIndex(unsigned long head) {
  /* Bits (15 to 12)
     see table above for full values */
  unsigned long mask = (1 << 15) | (1 << 14) | (1 << 13) | (1 << 12);
  int index = (head & mask) >> 12;
  if (index == 0xF) {
    //    cerr << "Error: reserved MPEG bitrate of 1111 detected" << endl;
    index = -1;
  }

  return index;
}

/** 
 * Returns the sampling frequency from the provided header.
 */
static int getMP3SampFrequency(unsigned long head) {
  /* Bits (11, 10)
     see table above for decode of the values.
     1,1 == reserved */
  unsigned long mask = (1 << 11) | (1 << 10);
  int freq = (head & mask) >> 10;
  if (freq == 0x3) {
    //    cerr << "Error: reserved MPEG frequency index of 11 detected" << endl;
    freq = -1;
  }

  return freq;
}

/* head_check is from mpg123 0.59r common.c */
static bool head_check(unsigned long head) {
  /* Bits (31 to 21) must all be set/ */
  if ( (head & 0xffe00000) != 0xffe00000 )
    return false;
  /* makes sure the mp3 layer is not "00 - reserved" 
     (right now the Rio only supports Layer III MPEG's) */
  if ( ((head>>17)&3) != MPEG_LAYER_III )     
    return false;                  
  /* check to make sure that the bit rate is correct. */
  /* "1111 - reserved is not accepted" */
  if ( ((head>>12)&0xf) == 0xf ) 
    return false;                
  /* Make sure the sampling freq. index is not "11 - reserved" */
  if ( ((head>>10)&0x3) == 0x3 ) 
    return false;
  if ( (head & 0xffff0000) == 0xfffe0000 )
    return false;

  return true;
}

/*
  rioInstance::MP3getInfo:
    Function takes in a file name (MP3) and returns a
  Info structure containing the amount of junk (in bytes)
  and a compete Rio header struct.
*/
Info rioInstance::MP3getInfo(const char *file_name){
  unsigned char buf[128];
  rio_file *mp3_file;
  Info newInfo;
  int i, ret;
  struct stat statinfo;
  char *tmp1, *tmp2;
  
#if HAVE_LIBID3
  ID3Tag *id3;
  ID3Frame *frame;
  ID3Field *field;
#endif  

  bzero(buf, 128);

  if (stat(file_name, &statinfo) < 0){
    newInfo.data = NULL;
    return newInfo;
  }

  mp3_file = new rio_file;
  bzero(mp3_file, sizeof(rio_file));

  mp3_file->size = statinfo.st_size;
  
  /* chop the file down to the last component,
     and remove the trailing .mp3 if it's there */
  
  tmp1 = new char[strlen(file_name) + 1];
  bzero(tmp1, strlen(file_name) + 1);
  
  strncpy(tmp1, file_name, strlen(file_name));
  tmp2 = basename(tmp1);
  
  for (i = strlen(tmp2) ; i > 0 && tmp2[i] != '.' ; i--);
  
  
  strncpy((char *)mp3_file->name , tmp2, 63);
  if (i > 0) tmp2[i] = '\0';
  strncpy((char *)mp3_file->title, tmp2, 63);

  delete[] tmp1;
  
  /* for those of you who can get libid3 working */
#ifdef HAVE_LIBID3
  if ((id3 = ID3Tag_New()) == NULL){
    printf("Error in ID3Tag_New()\n");
  }
  
  ID3Tag_Link(id3, file_name);
  if ((frame = ID3Tag_FindFrameWithID(id3, ID3FID_TITLE)) != NULL){
    if ((field = ID3Frame_GetField(frame, ID3FN_TEXT)) != NULL){
#if defined(ID3_4ARGS)
      ID3Field_GetASCII(field, (char *)mp3_file->title, 1024, 1);
#else
      ID3Field_GetASCII(field, (char *)mp3_file->title, 1024);
#endif
      if (output){
	output->form("Title: %s", mp3_file->title);
	*output<<endl;
      }
    }
  }
  
  if ((frame = ID3Tag_FindFrameWithID(id3, ID3FID_ALBUM)) != NULL){
    if ((field = ID3Frame_GetField(frame, ID3FN_TEXT)) != NULL){
#if defined(ID3_4ARGS)
      ID3Field_GetASCII(field, (char *)mp3_file->album, 1024, 1);
#else
      ID3Field_GetASCII(field, (char *)mp3_file->album, 1024);
#endif
      if (output){
	output->form("Album: %s", mp3_file->album);
	*output<<endl;
      }
    }
  }
  
  if ((frame = ID3Tag_FindFrameWithID(id3, ID3FID_LEADARTIST)) != NULL){
    if ((field = ID3Frame_GetField(frame, ID3FN_TEXT)) != NULL){
#if defined(ID3_4ARGS)
      ID3Field_GetASCII(field, (char *)mp3_file->artist, 1024, 1);
#else
      ID3Field_GetASCII(field, (char *)mp3_file->artist, 1024);
#endif
      if (output){
	output->form("Artist: %s\n", mp3_file->artist);
	*output<<endl;
      }
    }
  }
#endif
  
  long int head;
  int fd;
  if ((fd = open (file_name, O_RDONLY)) < 0){
    delete mp3_file;
    newInfo.data = NULL;
    return newInfo;
  }

  /* got this from mpg123 as well */
  for (i = 0 ; i < 65536 ; i++){
    read(fd, &head, 4);
#if BYTE_ORDER == LITTLE_ENDIAN
    head = bswap_32(head);
#endif

    if (head_check(head))
      break;
    else
      lseek(fd, -3, SEEK_CUR);  
  }

  if (i == 65536){
    delete mp3_file;
    newInfo.data = NULL;
    return newInfo;
  }else
    newInfo.junk = i;

  /* the file that will be uploaded is smaller if there is junk */
  mp3_file->size -= i;

  int freq_index    = getMP3SampFrequency(head);
  int layer         = getMP3Layer(head);
  int version       = getMPEGVersion(head);
  int bitrate_index = getMP3BitrateIndex(head);

  /* Make sure that the values we extracted are correct. All routines return -1
     on error and print an error message so all we have to do is bail out. */
  if (freq_index == -1 || layer == -1 || 
      version == -1 || bitrate_index == -1) {
    delete mp3_file;
    newInfo.data = NULL;
    return newInfo;
  }

  if ( version == MPEG_VERSION_1 ) {
    mp3_file->bit_rate = MP3_BITRATES[0][3 - layer][bitrate_index] << 7;
  } else {
    mp3_file->bit_rate = MP3_BITRATES[1][3 - layer][bitrate_index] << 7;
  }
  
  mp3_file->sample_rate =  MP3_FREQS[version][freq_index];
  /* simplified from (1152*size)/(144000*bitrate) so as not to run over long int */
  mp3_file->time = (long int)((double)mp3_file->size/
			      ((double)(mp3_file->bit_rate >> 7) * 125));

  if (output){
    output->form("Bitrate: %03i  Samplerate: %05i", mp3_file->bit_rate >> 7, mp3_file->sample_rate);
    *output<<endl;
  }

  /* it is an mp3 all right, finish up the INFO structure */
  mp3_file->mod_date = time(NULL);
  mp3_file->bits = 0x10000b11;
  mp3_file->type = TYPE_MP3;
  mp3_file->foo4 = 0x00020000;

  newInfo.data = mp3_file;

  close(fd);

  return newInfo;
}
