#include "config.h"
#include "rio-cpp.h"
#include "main.h"

#include <string>
#include <iostream>

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>

#include <sys/stat.h>

using namespace std;

int main(int argc, char *argv[]){
  int c, option_index, ret = 0;
  int aflag = 0, dflag = 0, uflag = 0, nflag = 0, dev = 0, elvl = 0, bflag = 0;
  char *aopt = NULL, *uopt = NULL;
  int dopt = 0, copt = 0;
  char *title = NULL, *artist = NULL, *album = NULL, *name = NULL;

  int lflag = 0, hflag = 0, iflag = 0, vflag = 0, fflag = 0, cflag = 0;

  struct option long_options[] = {
    {"upload",  1, 0, 'a'},
    {"bulk",    0, 0, 'b'},
    {"delete",  1, 0, 'd'},
    {"update",  1, 0, 'u'},
    {"artist",  1, 0, 's'},
    {"title" ,  1, 0, 't'},
    {"album" ,  1, 0, 'r'},
    {"name",    1, 0, 'n'},
    {"device",  1, 0, 'o'},
    {"debug",   1, 0, 'e'},
    {"download",1, 0, 'c'},
    {"format",  0, 0, 'f'},
    {"list",    0, 0, 'l'},
    {"help",    0, 0, 'h'},
    {"info",    0, 0, 'i'},
    {"version", 0, 0, 'v'}
  };

  while((c = getopt_long(argc, argv, "W;a:bld:e:c:u:s:t:r:o:n:fh?iv",
			 long_options, &option_index)) != -1){

    switch(c){
    case 'a':
      aflag = 1;
      aopt = optarg;

      break;
    case 'b':
      aflag = 1;
      bflag = 1;
      
      break;
    case 'l':
      lflag = 1;

      break;
    case 'd':
      dflag = 1;
      dopt = atoi(optarg);

      break;
    case 'e':
      elvl = atoi(optarg);

      break;
    case 'c':
      cflag = 1;
      copt = atoi(optarg);

      break;
    case 'u':
      uflag = 1;
      uopt = optarg;

      break;
    case 't':
      title = optarg;

      break;
    case 'r':
      album = optarg;

      break;
    case 'o':
      dev = atoi(optarg);

      break;
    case 'n':
      nflag = 1;
      name = optarg;

      break;
    case 's':
      artist = optarg;

      break;
    case 'f':
      fflag = 1;

      break;
    case 'h':
    case '?':
      hflag = 1;

      break;
    case 'i':
      iflag = 1;

      break;
    case 'v':
      vflag = 1;

      break;
    default:
      cout<<"Unknown option -"<<(char)c<<"."<<endl;
      usage();

      return -1;
    }
  }

  if (vflag){
    version();

    return 0;
  }

  if (!aflag && !dflag && !vflag && !uflag && !fflag && !iflag && !lflag && !nflag && !cflag){
    usage();
    exit(1);
  }

  rioInstance *currentRio;
  if (dev >= 0 && dev < 16)
    if (!elvl){
      currentRio = new rioInstance(dev, &cout);
    }else
      currentRio = new rioInstance(dev, &cout, &cerr, elvl);
  else
    currentRio = new rioInstance;


  if (currentRio->exists() == false){
    cout<<"No rio found."<<endl;
#if defined (WITH_USBDEVFS)
    cout<<"If the rio kernel module is loaded or if you do not have usbdevfs support in your kernel"<<endl;
    cout<<"this version will not work. Try compiling rioutil without usbdevfs."<<endl;
#elif defined (__MacOSX__)
    cout<<"Make sure that your rio is plugged in and make sure you have no other programs running"<<endl;
    cout<<"that could be using it (such as iTunes)."<<endl;
#else
    cout<<"Please make sure your rio is plugged in and that the kernel module is loaded"<<endl;
#endif

    delete currentRio;
    return 0;
  }

  currentRio->setProgress(progress, NULL);

  if (cflag){
    currentRio->download(copt);

    delete currentRio;
    return 0;
  }

  if (nflag){
    currentRio->setName(name);

    delete currentRio;
    return 0;
  }

  if (iflag){
    int total = currentRio->returnFree() + currentRio->returnUsage();
    // the 1024s are just to prevent overflow
    long int ticks = 50 * (currentRio->returnUsage()/1024) / (total/1024);
    cout<<"Name: "<<currentRio->returnName()<<endl;
    cout.form("Free: %03.01f MB (", (double)currentRio->returnFree() / 1048576.0);
    cout.form("%03.01f MB Total)", 
	   (double)(currentRio->returnFree() + currentRio->returnUsage()) / 1048576.0);
    cout<<endl;
    cout<<"[";
    for(int i = 1; i<=50 ; i++){
      if (i <= ticks)
	cout.put('#');
      else
	cout.put(' ');
    }
    cout.form("] %03.01f %%", 
	   100 * (double(currentRio->returnUsage()) / double(total))); 
    cout<<endl<<endl;
    long int ttime = currentRio->returnTime();
    cout.form("Total Time: %02i:%02i:%02i", int(ttime) / 3600, 
	   int(ttime % 3600) / 60, int(ttime) % 60);
    cout<<endl<<endl;
    cout<<"Volume: "<<currentRio->returnVolume()<<endl;
    cout<<"Repeat: "<<repeatStates[currentRio->returnRepeatState()]<<endl;
    cout<<"Equilizer: "<<equilizerStates[currentRio->returnEquilizerState()]<<endl;
    if (currentRio->returnType() == rioInstance::RIO800)
      cout<<"Contrast: "<<currentRio->returnContrast()<<endl;
    cout<<"Backlight: "<<lightStates[currentRio->returnLightState()]<<endl;
    cout<<"Programmed to Playlist: "<<currentRio->returnPlaylist()<<endl<<endl;

    cout<<"Firmware Version: "<<currentRio->returnFirm()<<endl;

    if (lflag)
      cout<<endl;
  }

  if (lflag){
    cout.form("%2s %2s %s | %36s | %36s  %32s   %2s:%2s %5s %6s %3s\n",
	      "id",
	      "No", "d", "Artist", "Title", "Name", "mm", "ss", "Size", "SR"
	      , "BR");
    for (int i = 0; i<145 ; i++)
      cout.put('-');
    
    cout<<endl;
    currentRio->iterate(printfiles);
  }

  if (fflag){
    if (currentRio->format() == 0)
      cout<<"Rio memory format complete."<<endl;
    else
      cout<<"Rio memory format did not complete."<<endl;
    
    currentRio->refreshFileList();
  }

  if (uflag){
    if (currentRio->update(uopt) == 0)
      cout<<"Rio update completed successfully."<<endl;
    else
      cout<<"Rio update was not completed successfully."<<endl;
      
    currentRio->refreshFileList();

    delete currentRio;
    return 0;
  }

  if (dflag){
    if (currentRio->delete_file(dopt) == 0)
      cout<<"File "<<dopt<<" deleted successfully."<<endl;
    else
      cout<<"File "<<dopt<<" could not be deleted."<<endl;
    
    currentRio->refreshFileList();

    delete currentRio;
    return 0;
  }

  if (aflag){
    cout.form("Free space is %03.01f MB.", 
	      currentRio->returnFree() / 1048576.0);
    cout<<endl;

    if (bflag)
      currentRio->upload(&argv[optind - 1], artist, album);
    else{
      int i = optind - 1;
      aopt = argv[i];
      cout<<"Uploading "<<aopt<<"."<<endl;
      
      struct stat statinfo;
      if (stat(aopt, &statinfo) < 0){
	cout<<"Could not stat "<<aopt<<"!"<<endl;
	exit(1);
      }else{
	cout.form("File size is %03.01f MB.", 
		  (double)statinfo.st_size / 1048576.0);
	cout<<endl;
      }
      
      if ((ret = currentRio->upload(aopt, artist, title)) == 0)
	cout<<"File "<<aopt<<" uploaded."<<endl;
      else if (ret == ERIOFULL){
	cout<<"File "<<aopt<<" could not be uploaded."<<endl;
	cout<<"Reason Given: Not enough free space."<<endl;
      }else
	cout<<"File "<<aopt<<" could not be uploaded."<<endl;
    }
  }

  delete currentRio;
  return 0;
}
  
void printfiles(rio_file *file){
  rio_file *tmpf;
  static int idx = 0;

#if BYTE_ORDER == BIG_ENDIAN
  tmpf = swap_file(file);
#else
  tmpf = new rio_file;
  *tmpf = *file;
#endif

  printf("%02i %02i %i | %36s | %36s  %32s | %2i:%02i %5i %2i.%03i %03i\n",
	 idx++,
	 tmpf->file_no, 
	 (tmpf->bits & 0x80) ? 1 : 0,
	 tmpf->artist, tmpf->title,
	 tmpf->name, (tmpf->time / 60), (tmpf->time % 60),
	 tmpf->size / 1024, tmpf->sample_rate / 1000,
	 tmpf->sample_rate % 1000,
	 tmpf->bit_rate >> 7);

  delete tmpf;
}

void usage(void){
  cout<<"Usage: rioutil <OPTIONS>"<<endl;
  cout<<"Interface with Diamond MM MP3 players."<<endl;
  cout<<"An option is required, and if you use either the -t or"<<endl;
  cout<<"the -s options you must use them with the -a option"<<endl;
  cout<<endl;
  cout<<"  -o, --device=<int>     minor number of rio (assigned by driver), /dev/usr/rio?"<<endl;
  cout<<"  -l, --list             list files contained on the PDMP"<<endl;
  cout<<"  -f, --format           format rio internal memory"<<endl;
  cout<<"  -s, --artist=<string>  name of artist of song to be uploaded"<<endl;
  cout<<"  -t, --title=<string>   title of song to be uploaded"<<endl;
  cout<<"  -r, --album=<string>   album"<<endl;
  cout<<"  -n, --name=<string>    name rio"<<endl;
  cout<<"  -a, --upload=<file>    upload a file"<<endl;
  cout<<"  -b, --bulk             upload several files"<<endl;
  cout<<"  -d, --delete=<int>     delete a song from the rio (0 through ?)"<<endl;
  cout<<"  -e, --debug=<int>      set debug level"<<endl;
  cout<<"  -u, --update=<file>    update rio with a new firmware"<<endl;
  cout<<"  -?, --help             print this screen"<<endl<<endl;

  cout<<"This version was compiled for use with ";
#if defined (WITH_USBDEVFS)
  cout<<"usbdevfs. Make sure that usbdevfs ";
  cout<<"is mounted on /proc/bus/usb."<<endl;
  cout<<"You can mount usbdevfs with the command: mount -t usbdevfs none /proc/bus/usb."<<endl;
#elif defined (WITH_LIBUSB)
  cout<<"libusb.  Make sure that usbdevfs ";
  cout<<"is mounted on /proc/bus/usb."<<endl;
  cout<<"You can mount usbdevfs with the command: mount -t usbdevfs none /proc/bus/usb."<<endl;
#elif defined (__MacOSX__)
  cout<<"MacOS X. Make sure no other programs are using the rio."<<endl;
#else
  cout<<"kernel level drivers. Make sure that the driver"<<endl;
  cout<<"is built and loaded."<<endl;
#endif
}

#define TOTAL_MARKS  50

void progress(int x, int X, void *){
  int nummarks = (x * TOTAL_MARKS) / X;
  int percent = (x * 100) / X;
  char m[] = "-\\|/";
  char HASH_MARK;

  if (percent != 100)
    HASH_MARK  = '-';
  else
    HASH_MARK  = '*';

  char HASH_BARRIER = '>';
  char NO_HASH      = ' ';

  cout.form("%c [[34m", m[percent%4]);
  
  for (int i = 0 ; i < TOTAL_MARKS ; i++){
    if (i < nummarks)
      cout.put(HASH_MARK);
    else if (i == nummarks)
      cout.put(HASH_BARRIER);
    else
      cout.put(NO_HASH);
  }

  cout.form("[m] [37m%3i[m%%", percent);

  if (x == X)
    cout<<endl;
  else
    for(int i = 0 ; i < (TOTAL_MARKS + 9) ; i++) cout.put('\b');

  cout.flush();
}

void version(void){
  cout<<PACKAGE<<" "<<VERSION<<endl;
  cout<<"Copyright (C) 2001 Nathan Hjelm"<<endl;
  cout<<endl;
  cout<<PACKAGE<<" comes with NO WARRANTY, to the extent permitted by law."<<endl;
  cout<<"You may redistribute copies of "<<PACKAGE<<" under the terms"<<endl;
  cout<<"of the GNU Lesser Public License."<<endl;
  cout<<"For more information about these issues"<<endl;
  cout<<"see the file named COPYING in the "<<PACKAGE<<" distribution."<<endl;
}
