
#include <stdio.h>
#include <rio-cwrapper.h>
#include "if_main.h"
#include "if_error.h"
#include "if.h"
#include "rio.h"

static crioInstance rioInstance = NULL;
static guint msg_statusbar_id = 0;
static int rio_nmb_files = 0;
static int rio_selected_file = -1;

#define MSG_NO_RIO				"No Rio player"
#define MSG_SEARCH_RIO		"Searching for a Rio player... Please wait..."
#define MSG_FILE_LIST			"Getting Rio content... Please wait..."
#define MSG_FILE_DELETE		"Deleting file %d... Please wait..."

static const char *unknown = "Unknown";

#define NMB_EQUALIZER_STATES 8
static const char *equalizerStates[NMB_EQUALIZER_STATES] = {
  "Normal", "Jazz", "Rock", "Classic",
  "Book", "Rap", "Flat", "Custom"
};

#define NMB_LIGHT_STATES 6
static const char *lightStates[NMB_LIGHT_STATES] = {
  "Always off", "1 sec", "2 secs",
  "5 secs", "10 secs", "Always on"
};
  
#define NMB_REPEAT_STATES 4
static const char *repeatStates[NMB_REPEAT_STATES] = {
  "Off", "All", "Track", "Playlist"
};

void rio_open()
{
  int debugLevel = 0;
	long int totalTime = 0;
	char infos[80];

	if (rioInstance)
		return;

	/* Display the searching box */
	msg_statusbar_id = if_main_statusbar_push(MSG_SEARCH_RIO);

	/* Treat pending events to display box */
	if_treat_pending_events();

  rioInstance = init_instance(rioInstance, 0, debugLevel, 0, 0);

	/* Suppress the searching box */
	if_main_statusbar_pop(msg_statusbar_id);

  if (! rioInstance)
	{
		msg_statusbar_id = if_main_statusbar_push(MSG_NO_RIO);
		menu_sync();
    return;
	}

  if (! exists(rioInstance))
  {
		msg_statusbar_id = if_main_statusbar_push(MSG_NO_RIO);
		rio_close();
		menu_sync();
    return;
  }

	if ((return_name(rioInstance) == NULL) || (return_version(rioInstance) == 0))
  {
		msg_statusbar_id = if_main_statusbar_push(MSG_NO_RIO);
		rio_close();
		menu_sync();
    return;
  }

	totalTime = return_time(rioInstance);

	sprintf(infos, "%s (%d Mb) - occupied: %d Mb (%d:%02d)",
		return_name(rioInstance),
		(return_usage(rioInstance)+return_free(rioInstance))/(1024*1024),
		return_usage(rioInstance)/(1024*1024),
		(int)(totalTime/60), (int)(totalTime%60)
	);
	msg_statusbar_id = if_main_statusbar_push(infos);

	rio_list();
	menu_sync();
}

void rio_close()
{
	if (!rioInstance)
		return;
	delete_instance(rioInstance);
	rioInstance = NULL;
	rio_nmb_files = 0;
	rio_selected_file = -1;
	menu_sync();
	clear_main_list();
	if_main_statusbar_pop(msg_statusbar_id);
	msg_statusbar_id = if_main_statusbar_push(MSG_NO_RIO);
}



void handle_file_info(rio_file *data)
{
	char id[8];
	char duration[8];
	char size[8];
	char br[8];
	char sr[8];
	char type[6];

	switch (data->type)
	{
		case 0x50524546: /* PREF */
			strcpy(type, "PREF");
			break;

		case 0x3347504d:
			strcpy(type, "MP3");
			break;

		case 0x20414d57:
			strcpy(type, "WMA");
			break;

		case 0x504c4341:
		case 0x45564157:
			strcpy(type, "WAV");
			break;

		default:
			strcpy(type, "???");
	}

	sprintf(id, "%d", data->file_no);
	sprintf(duration, "%d:%02d", (data->time)/60, (data->time)%60);
	if (data->size < 1024)
		sprintf(size, "%d", data->size);
	else if (data->size < 1024*1024)
		sprintf(size, "%.2f Kb", ((double)data->size)/((double)1024));
	else
		sprintf(size, "%.2f Mb", ((double)data->size)/((double)1024*1024));
	sprintf(br, "%d", (data->bit_rate)>>7);
	sprintf(sr, "%.1f", ((double)data->sample_rate)/((double)1000));

	add_file_to_main_list(
		id,
		data->title,
		data->artist,
		duration,
		size,
		br,
		sr,
		type,
		data->name
	);
	rio_nmb_files++;
}

void rio_list()
{
	int statusbar_id;

	statusbar_id = if_main_statusbar_push(MSG_FILE_LIST);
	refresh_files(rioInstance);
	clear_main_list();
	rio_nmb_files = 0;
	iterate_files(rioInstance, handle_file_info);
	if_main_statusbar_pop(statusbar_id);
}

double rio_get_version()
{
	if (!rioInstance)
		return 0;

	return return_version(rioInstance);
}

int rio_get_type()
{
	if (!rioInstance)
		return 0;

	return return_type(rioInstance);
}

char *rio_get_name()
{
	if (!rioInstance)
		return NULL;

	return return_name(rioInstance);
}

int rio_get_num_memdev()
{
	if (!rioInstance)
		return 0;

	return return_num_memdev(rioInstance);
}

int rio_get_usage()
{
	if (!rioInstance)
		return 0;

	return return_usage(rioInstance);
}

int rio_get_free()
{
	if (!rioInstance)
		return 0;

	return return_free(rioInstance);
}

long int rio_get_time()
{
	if (!rioInstance)
		return 0;

	return return_time(rioInstance);
}

int rio_get_volume()
{
	if (!rioInstance)
		return 0;

	return return_volume(rioInstance);
}

int rio_get_eq_state()
{
	if (!rioInstance)
		return 0;

	return return_eq_state(rioInstance);
}

const char *rio_get_eq_state_name()
{
	int state;

	if (!rioInstance)
		return NULL;

	state = return_eq_state(rioInstance);
	if (state < NMB_EQUALIZER_STATES)
		return equalizerStates[state];

	return unknown;
}

int rio_get_light_state()
{
	if (!rioInstance)
		return 0;

	return return_light_state(rioInstance);
}

const char *rio_get_light_state_name()
{
	int state;

	if (!rioInstance)
		return NULL;

	state = return_light_state(rioInstance);
	if (state < NMB_LIGHT_STATES)
		return lightStates[state];

	return unknown;
}

int rio_get_rep_state()
{
	if (!rioInstance)
		return 0;

	return return_rep_state(rioInstance);
}

const char *rio_get_rep_state_name()
{
	int state;

	if (!rioInstance)
		return NULL;

	state = return_rep_state(rioInstance);
	if (state < NMB_REPEAT_STATES)
		return repeatStates[state];

	return unknown;
}

int rio_get_contrast()
{
	if (!rioInstance)
		return 0;

	return return_contrast(rioInstance);
}

int rio_get_playlist()
{
	if (!rioInstance)
		return 0;

	return return_playlist(rioInstance);
}

int rio_get_nmb_files()
{
	return rio_nmb_files;
}

void *rio_get_instance()
{
	return rioInstance;
}

void rio_set_selected_file(int id)
{
	rio_selected_file = id;
}

int rio_get_selected_file()
{
	return rio_selected_file;
}

int rio_delete_file(int id)
{
	int error;
	int statusbar_id;
	char buf[128];

	if (!rioInstance)
		return 1;

	snprintf(buf, 128, MSG_FILE_DELETE, id);
	statusbar_id = if_main_statusbar_push(buf);
	error = delete_file(rioInstance, id-1);

	if_main_statusbar_pop(statusbar_id);

	return error;
}
